/**
 * VRAM
 * 
 * MPLAB X IDE v5.30 with XC8 Ver2.30
 * (c)2020 Futoshi Sumikawa. 2020/11/20
 */
/******************************************************************************/
/* Files to Include                                                           */
/******************************************************************************/

#include <xc.h>         /* XC8 General Include File */
#include <stdint.h>         /* For uint8_t definition */
#include <stdbool.h>
#include <stdint.h>
#include <stdint.h>
#include <stdlib.h>
#include <stdlib.h>        /* For true/false definition */

#include "vram.h"
#include "font.h"

uint8_t _vramColor = 3;
int8_t _vramPositionX = 0;
int8_t _vramPositionY = 0;
uint8_t _vramFont = 1;

// ---------------------------------------
// VRAM
//         C1  C2  C3  C4  C5  C6  C7  C8
//        +00             +01             
// R1A 00 [xx][xx][xx][xx][xx][xx][xx][xx]
// R2A 02 [xx][xx][xx][xx][xx][xx][xx][xx]
// R3A 04 [xx][xx][xx][xx][xx][xx][xx][xx]
// R4A 06 [xx][xx][xx][xx][xx][xx][xx][xx]
// R5A 08 [xx][xx][xx][xx][xx][xx][xx][xx]
// R6A 0a [xx][xx][xx][xx][xx][xx][xx][xx]
// R7A 0c [xx][xx][xx][xx][xx][xx][xx][xx]
// R8A 0e [xx][xx][xx][xx][xx][xx][xx][xx]
//
// R1B 10 [xx][xx][xx][xx][xx][xx][xx][xx]
// R2B 12 [xx][xx][xx][xx][xx][xx][xx][xx]
// R3B 14 [xx][xx][xx][xx][xx][xx][xx][xx]
// R4B 16 [xx][xx][xx][xx][xx][xx][xx][xx]
// R5B 18 [xx][xx][xx][xx][xx][xx][xx][xx]
// R6B 1a [xx][xx][xx][xx][xx][xx][xx][xx]
// RB7 1c [xx][xx][xx][xx][xx][xx][xx][xx]
// RB8 1e [xx][xx][xx][xx][xx][xx][xx][xx]

// VRAM buffer
//         C1  C2  C3  C4  C5  C6  C7  C8
//        +00 +01 +02 +03 +04 +05 +06 +07
// R1A 00 [xx][xx][xx][xx][xx][xx][xx][xx]
// R2A 08 [xx][xx][xx][xx][xx][xx][xx][xx]
// R3A 10 [xx][xx][xx][xx][xx][xx][xx][xx]
// R4A 18 [xx][xx][xx][xx][xx][xx][xx][xx]
// R5A 20 [xx][xx][xx][xx][xx][xx][xx][xx]
// R6A 28 [xx][xx][xx][xx][xx][xx][xx][xx]
// R7A 30 [xx][xx][xx][xx][xx][xx][xx][xx]
// R8A 38 [xx][xx][xx][xx][xx][xx][xx][xx]
//
// R1B 40 [xx][xx][xx][xx][xx][xx][xx][xx]
// R2B 48 [xx][xx][xx][xx][xx][xx][xx][xx]
// R3B 50 [xx][xx][xx][xx][xx][xx][xx][xx]
// R4B 58 [xx][xx][xx][xx][xx][xx][xx][xx]
// R5B 60 [xx][xx][xx][xx][xx][xx][xx][xx]
// R6B 68 [xx][xx][xx][xx][xx][xx][xx][xx]
// R7B 70 [xx][xx][xx][xx][xx][xx][xx][xx]
// R8B 78 [xx][xx][xx][xx][xx][xx][xx][xx]

uint8_t _vramBuffer[128];

uint8_t _vram[32] = {
    0b00000000, 0b00000000,
    0b01010101, 0b01010101,
    0b10101010, 0b10101010,
    0b11111111, 0b11111111,
    0b11100100, 0b11100100,
    0b11100100, 0b11100100,
    0b11100100, 0b11100100,
    0b11100100, 0b11100100,

    0b00000000, 0b00000000,
    0b01010101, 0b01010101,
    0b10101010, 0b10101010,
    0b11111111, 0b11111111,
    0b11100100, 0b11100100,
    0b11100100, 0b11100100,
    0b11100100, 0b11100100,
    0b11100100, 0b11100100
};

uint8_t vramPeekBuffer(uint8_t address) {
    return _vramBuffer[address]  & 0b00000011;
}

void vramInvalidate(void) {
    uint8_t address = 0;
    for (uint8_t i = 0; i < 32; i++) {
        uint8_t data = _vram[i];
        _vramBuffer[address + 3] = (data & 0b00000011);
        data = data >> 2;
        _vramBuffer[address + 2] = (data & 0b00000011);
        data = data >> 2;
        _vramBuffer[address + 1] = (data & 0b00000011);
        data = data >> 2;
        _vramBuffer[address] = data;
        address+=4;
    }
}

void vramClear(void) {
    for (uint8_t i = 0; i < 32; i++) {
        _vram[i] = 0;
    }
}

void vramSetColor(uint8_t color) {
    _vramColor = color;
}

void vramSetPosition(int8_t x, int8_t y) {
    _vramPositionX = x;
    _vramPositionY = y;
}

void vramSetFont(uint8_t font) {
    _vramFont = font;
}

uint8_t vramPget(uint8_t x, uint8_t y) {
    uint8_t color = 0;
    
    if (x < 16 && y < 8) {
 
        uint8_t address = 0;
        if (x < 8) {
            address = y * 2 + x / 4;
        } else {
            x = x - 8;
            address = 16 + y * 2 + x / 4;
        }

        uint8_t v = _vram[address];
        if (x == 0 || x == 4) {
            color = (uint8_t)(v & 0b11000000) >> 6;
        } else if (x == 1 || x == 5) {
            color = (uint8_t)(v & 0b00110000) >> 4;
        } else if (x == 2 || x == 6) {
            color = (uint8_t)(v & 0b00001100) >> 2;
        } else if (x == 3 || x == 7) {
            color = (uint8_t)(v & 0b00000011);
        }
    }
    
    return color;
}

/**
 * PSET
 * @param x 0 - 16
 * @param y 0 - 8
 * @param c 0 - 3
 * @param xor 0 or 1
 */
void vramPset(uint8_t x, uint8_t y, uint8_t color, uint8_t xor) {
    
    if (x < 16 && y < 8) {
        
        color = color & 0b00000011;

        uint8_t address = 0;
        if (x < 8) {
            address = y * 2 + x / 4;
        } else {
            x = x - 8;
            address = 16 + y * 2 + x / 4;
        }

        uint8_t v = _vram[address];
        if (x == 0 || x == 4) {
            if (xor == 0) {
                v = v & 0b00111111;
            }
            v = v ^ (uint8_t)(color << 6);
        } else if (x == 1 || x == 5) {
            if (xor == 0) {
              v = v & 0b11001111;
            }
            v = v ^ (uint8_t)(color << 4);
        } else if (x == 2 || x == 6) {
            if (xor == 0) {
              v = v & 0b11110011;
            }
            v = v ^ (uint8_t)(color << 2);
        } else if (x == 3 || x == 7) {
            if (xor == 0) {
               v = v & 0b11111100;
            }
            v = v ^ color;
        }
        _vram[address] = v;
    }
}

void vramPut(uint8_t code, uint8_t xor) {
    for (uint8_t x = 0; x < 8; x++) {

        uint8_t font = FONT_LED_3x7;
        if (_vramFont == FONT_LED_3x7) {
            font = _fontLed3x7_30[(code - 0x30) * 8 + x];
        } else if (_vramFont == FONT_MISAKI) {
            font = _fontMisakiGothic2_30[(code - 0x30) * 8 + x];
        } else if (_vramFont == FONT_HOURGLASS_LOWER) {
            if (code < 0x10) {
                font = _fontHourglass_lower_00[code * 8 + x];
            } else if (code < 0x20) {
                font = _fontHourglass_lower_10[(code - 0x10) * 8 + x];
            } else if (code < 0x30) {
                font = _fontHourglass_lower_20[(code - 0x20) * 8 + x];
            }
        } else if (_vramFont == FONT_HOURGLASS_UPPER) {
            if (code < 0x10) {
                font = _fontHourglass_upper_00[code * 8 + x];
            } else if (code < 0x20) {
                font = _fontHourglass_upper_10[(code - 0x10) * 8 + x];
            } else if (code < 0x30) {
                font = _fontHourglass_upper_20[(code - 0x20) * 8 + x];
            }
        } else if (_vramFont == FONT_CIRCLE) {
            font = _fontCircle_00[code * 8 + x];
        }

        uint8_t mask = 0b10000000;
        for (uint8_t y = 0; y < 8; y++) {
            if (font & mask) {
                vramPset((uint8_t)(x + _vramPositionX), (uint8_t)(7 - y + _vramPositionY), _vramColor, xor);
            }
            mask = mask >> 1;
        }
    }
}

void vramRotateLeft(void) {
    uint8_t w[64];
    
    // Rotate A to the left
    for (uint8_t x = 0; x < 8; x++) {
        for (uint8_t y = 0; y < 8; y++) {
            w[(7 - x) * 8 + y] = vramPget(x, y);
        }
    }
    // Clear A
    for (uint8_t i = 0; i < 16; i++) {
        _vram[i] = 0;
    }
    // Transfer work to A
    for (uint8_t x = 0; x < 8; x++) {
        for (uint8_t y = 0; y < 8; y++) {
            vramPset(x, y, w[y * 8 + x], 1);
        }
    }

    // Rotate B to the left
    for (uint8_t x = 0; x < 8; x++) {
        for (uint8_t y = 0; y < 8; y++) {
            w[(7 - x) * 8 + y] = vramPget(x + 8, y);
        }
    }
    // Clear B
    for (uint8_t i = 0; i < 16; i++) {
        _vram[i + 16] = 0;
    }
    // Transfer work to B
    for (uint8_t x = 0; x < 8; x++) {
        for (uint8_t y = 0; y < 8; y++) {
            vramPset(x + 8, y, w[y * 8 + x], 1);
        }
    }
}

void vramRotateRight(void) {
    uint8_t w[64];

    // Rotate A to the right
    for (uint8_t x = 0; x < 8; x++) {
        for (uint8_t y = 0; y < 8; y++) {
            w[x * 8 + 7 - y] = vramPget(x, y);
        }
    }
    // Clear A
    for (uint8_t i = 0; i < 16; i++) {
        _vram[i] = 0;
    }
    // Transfer work to A
    for (uint8_t x = 0; x < 8; x++) {
        for (uint8_t y = 0; y < 8; y++) {
            vramPset(x, y, w[y * 8 + x], 1);
        }
    }

    // Rotate B to the right
    for (uint8_t x = 0; x < 8; x++) {
        for (uint8_t y = 0; y < 8; y++) {
            w[x * 8 + 7 - y] = vramPget(x + 8, y);
        }
    }
    // Transfer A to B
    for (uint8_t i = 0; i < 16; i++) {
        _vram[i + 16] = _vram[i];
    }
    // Clear A
    for (uint8_t i = 0; i < 16; i++) {
        _vram[i] = 0;
    }
    // Transfer work to A
    for (uint8_t x = 0; x < 8; x++) {
        for (uint8_t y = 0; y < 8; y++) {
            vramPset(x, y, w[y * 8 + x], 1);
        }
    }
}

void vramCylinderEffect(void) {
    for (uint8_t n = 0; n < 2; n++) {
        for (uint8_t y = 0; y < 8; y++) {

            uint8_t address = y * 2 + n * 16;
            
            uint8_t vl = _vram[address];
            if ((vl & 0b11000000) != 0) {
                vl = (vl & 0b00111111) | 0b01000000;
            }
            if ((vl & 0b00110000) != 0) {
                vl = (vl & 0b11001111) | 0b00100000;
            }
            if ((vl & 0b00001100) != 0) {
                vl = (vl & 0b11110011) | 0b00001100;
            }
            if ((vl & 0b00000011) != 0) {
                vl = (vl & 0b11111100) | 0b00000011;
            }
            _vram[address] = vl;

            uint8_t vr = _vram[address + 1];
            if ((vr & 0b11000000) != 0) {
                vr = (vr & 0b00111111) | 0b11000000;
            }
            if ((vr & 0b00110000) != 0) {
                vr = (vr & 0b11001111) | 0b00110000;
            }
            if ((vr & 0b00001100) != 0) {
                vr = (vr & 0b11110011) | 0b00001000;
            }
            if ((vr & 0b00000011) != 0) {
                vr = (vr & 0b11111100) | 0b00000001;
            }
            _vram[address + 1] = vr;
        }
    }

    for (uint8_t x = 1; x < 7; x++) {
        if (vramPget(x, 6) != 0) {
            vramPset(x, 6, 2, 0); 
        }
        if (vramPget(x, 7) != 0) {
            vramPset(x, 7, 1, 0); 
        }
        if (vramPget(x + 8, 1) != 0) {
            vramPset(x + 8, 1, 2, 0); 
        }
        if (vramPget(x + 8, 0) != 0) {
            vramPset(x + 8, 0, 1, 0); 
        }
    }

    if (vramPget(1, 6) != 0) {
        vramPset(1, 6, 1, 0); 
    }
    if (vramPget(6, 6) != 0) {
        vramPset(6, 6, 1, 0); 
    }
    if (vramPget(9, 1) != 0) {
        vramPset(9, 1, 1, 0); 
    }
    if (vramPget(14, 1) != 0) {
        vramPset(14, 1, 1, 0); 
    }
    
}