//********************************************************************
//* HHE-123 Si5351 VFO with OLED SH1106
//* 2025/05/22 Thu Version 1.00 XIAO SAMD21(3.3V 48MHz)
//* 2025/06/12 Thu Version 1.10 XIAO SAMD21(3.3V 48MHz)
//*  If frequency is less than IF in LSB mode, change to USB mode
//* 2025/06/13 Fri Version 1.11 XIAO SAMD21(3.3V 48MHz)
//*  If frequency is less than IF in LSB mode, change to next mode
//*  Modification of mode setting
//*  CLK1(TX OSC) DRIVE changed from 2mA to 8mA
//*  Frequency and mode range check at TX
//*  Inspection whose frequency in LSB mode is less than or equal to IF
//* 2025/06/22 Sun Version 1.12 XIAO SAMD21(3.3V 48MHz)
//*  Modified LSB to upper superheterodyne
//*  Deleted => If frequency is less than IF in LSB mode, change to USB mode
//*  Maximum frequency changed to 220MHz
//*  100Hz, 50kHz and 100kHz added to STEP
//* 2025/06/27 Fri Version 1.13 XIAO SAMD21(3.3V 48MHz)
//*  CLK1(TX OSC) DRIVE changed from 8mA to 4mA
//* 2025/07/02 Wed Version 1.14 XIAO SAMD21(3.3V 48MHz)
//*  Upper side heterodyne
//*   LSB BFO 456.5kHz(+1.5kHz) modified
//*   USB BFO 453.5kHz(-1.5kHz) modified
//* 2025/07/03 Thu Version 1.15 XIAO SAMD21(3.3V 48MHz)
//*  Upper side heterodyne
//*   LSB: TX OSC + 453.5ｋHz、BFO 456.5kHz(+1.5kHz) modified
//*   USB: TX OSC + 456.5ｋHz、BFO 453.5kHz(-1.5kHz) modified
//* 2025/07/24 Thu Version 1.16 XIAO SAMD21(3.3V 48MHz)
//*   Move the start screen display to after setting Si5351
//* 2025/08/17 Sun Version 1.17 XIAO SAMD21(3.3V 48MHz)
//*  BFO modified
//*   LSB: TX OSC + 453.5ｋHz、BFO 453.5kHz(-1.5kHz) modified
//*   USB: TX OSC + 456.5ｋHz、BFO 456.5kHz(+1.5kHz) modified
//*  Adjust the sensitivity of Signal Meter A/D input
//*   1241 = 1V
//* 2025/08/25 Mon Version 1.18 XIAO SAMD21(3.3V 48MHz)
//*   Transmission frequency range change
//* Copyright (C) 2025
//*  by JG1CCL W3CCL Beard UCHIDA CCLab(Rabbit Hat Design)
//*
//* [References]
//*  https://projecthub.arduino.cc/CesarSound/10khz-to-225mhz-vforf-generator-with-si5351-version-2-acdc25
//*  https://github.com/garybourbier/-10kHz-to-225MHz-VFO-RF-Generator-with-Si5351-and-Arduino-
//*
//* Uses Brian Lowen Rotary Encoder Library
//*  https://github.com/brianlow/Rotary
//* Uses Etherkit Si5351 Library
//*  https://github.com/etherkit/Si5351Arduino
//* Uses Adafruit GFX Library
//*  https://github.com/adafruit/Adafruit-GFX-Library
//* Uses Adafruit BusIO Library
//*  https://github.com/adafruit/Adafruit_BusIO
//* Adafruit SSD1136 Library
//*  https://github.com/adafruit/Adafruit_SSD1306
//* Uses Adafruit Adafruit_SH110x
//*  https://github.com/adafruit/Adafruit_SH110X
//* Uses FlashStorage library
//*  https://github.com/cmaglie/FlashStorage
//*
//********************************************************************

//********************************************************************
//* Include（インクルード）
//********************************************************************
// Include required libraries
#include <SPI.h>
#include <Wire.h>
#include <Rotary.h>             // Brian Lowen Rotary Encoder Library
#include <si5351.h>             // Etherkit Si5351 Library
#include <Adafruit_GFX.h>       // Adafruit GFX Library
//#include <Adafruit_SSD1306.h>   // Adafruit SSD1136 Library
#include <Adafruit_SH110X.h>    // Adafruit Adafruit_SH110x
#include <FlashStorage.h>       // FlashStorage library

//********************************************************************
//* Definition（定義）
//********************************************************************
//#define DEBUG

// The pins for I2C are defined by the Wire-library
//  On an Arduino Pro Mini:          A4(SDA), A5(SCL)
//  On an Seeed Studio XIAO SAMD21:  A4(SDA), A5(SCL)

// Define Si5351 Module
//  Declaration for an Si5351 connected to I2C (SDA, SCL pins)
//  address: 96 (0x60) I2C-Address
//  Upper side heterodyne
//   SI5351_CLK0(SI5351_PLLA): RX BFO(When LSB 456.5kHz, When USB 453.5kHz, When CW 454.2kHz, Others 0Hz)
//  SI5351_CLK0(SI5351_PLLA): RX BFO(When LSB 453.5kHz, When USB 456.5kHz, When CW 454.2kHz, Others 0Hz)
//   2025/08/17 Sun Version 1.17 modified
//  SI5351_CLK1(SI5351_PLLA): TX OSC(10kHz to 220MHz) 2025/06/22 Sun Version 1.12 modified
//  SI5351_CLK2(SI5351_PLLB): RX LO(When 7MHz band or lower LSB, When 10MHz band or higher USB)
//   AM:  TX OSC + 455ｋHz、BFO 0Hz
//    LSB: TX OSC + 453.5ｋHz、BFO 456.5kHz(+1.5kHz) 2025/07/03 Thu Version 1.15 modified
//    USB: TX OSC + 456.5ｋHz、BFO 453.5kHz(-1.5kHz) 2025/07/03 Thu Version 1.15 modified
//   LSB: TX OSC + 453.5ｋHz、BFO 453.5kHz(-1.5kHz) 2025/08/17 Sun Version 1.17 modified
//   USB: TX OSC + 456.5ｋHz、BFO 456.5kHz(+1.5kHz) 2025/08/17 Sun Version 1.17 modified
//   CW:  TX OSC + 455ｋHz、BFO 454.2kHz(-800Hz)
#define FREQ_MIN        10000     // 10kHz
#define FREQ_MAX        220000000 // 220MHz 2025/06/22 Sun Version 1.12 modified
#define IF              455000    // Intermediate frequency（中間周波数）
  // 455000 = 455kHz, 10700000 = 10.7MHz, 0 = direct convert receiver or RF generator
#define AM_BFO          0         // BFO 0Hz
//#define LSB_BFO         456500    // BFO 456.5kHz(+1.5kHz)  2025/07/02 Wed Version 1.14 modified
//#define USB_BFO         453500    // BFO 453.5kHz(-1.5kHz)  2025/07/02 Wed Version 1.14 modified
#define LSB_BFO         453500    // BFO 453.5kHz(-1.5kHz)  2025/08/17 Sun Version 1.17 modified
#define USB_BFO         456500    // BFO 456.5kHz(+1.5kHz)  2025/08/17 Sun Version 1.17 modified
#define CW_BFO          454200    // BFO 454.2kHz(-800Hz)
#define BANDWIDTH       3000      // SSB bandwidth(3kHz)  2025/07/03 Thu Version 1.15 modifie
#define SI5351_CAL_F    1580      // Si5351 module calibration factor
#define BAND_6M         7         // Band plan table(JARL band plan 20230925) 6m value
#define BAND_INIT       BAND_6M   // Initial Band
#define STEP_INIT       5         // Initial Step 2025/06/22 Sun Version 1.12 modified
#define MODE_AM         0         // Mode tae AM value
#define MODE_LSB        1         // Mode table LSB valuebl
#define MODE_INIT       MODE_AM   // Initial Mode(AM, LSB, USB, CW)
#define VFO_MODE        0         // VFO Mode(Arguments used in the layout function)
#define MENU_MODE       1         // Menu Mode(Arguments used in the layout function)
#define STATUS_RX       0         // RX
#define STATUS_TX       1         // TX
#define STATUS_OUT_TX   2         // TX out of range
#define STATUS_OUT_MODE 3         // MODE out of range

// Define 0.96inch 128*64dots OLED(SSD1306) Module
//  Declaration for an SSD1306 display connected to I2C(SDA, SCL pins)
//  address: 60 (0x3C) I2C-Address
// Define 1.3inch 128*64dots OLED(SH1106) Module
//  Declaration for an SH1106 display connected to I2C(SDA, SCK pins)
//  address: 60 (0x3C) I2C-Address
#define SCREEN_WIDTH    128   // OLED display width, in pixels
#define SCREEN_HEIGHT   64    // OLED display height, in pixels
#define OLED_RESET      -1    // Reset pin #(or -1 if sharing Arduino reset pin)
#define SCREEN_ADDRESS  0x3C  // See datasheet for address: 0x3C or 0x3D
#define WHITE SH110X_WHITE    // Consistency of SSD1306.h and SH110X.h

// PORTs assign
// Rotary encoder
//  Rotary.h Pin Definitions
//   Hardware connections :
//    encoder DATA(A) pin to Arduino pin D3
//    encoder CLOCK(B) pin to Arduino pin D2
//    encoder SW pin to Arduino pin D4
//    encoder ground pin to ground(GND)
//    encoder 5V pin to 5V
//  Values returned by 'process'
//   No complete step yet
//    #define DIR_NONE 0x0
//   Clockwise step
//    #define DIR_CW 0x10
//   Counter-clockwise step
//    #define DIR_CCW 0x20
#define ENCODER_CLOCK_PIN 2
#define ENCODER_DATA_PIN  3
//#define ENCODER_SW_PIN    4

// I/O Pin Definition
#define STEP    7   // The pin used by tune step push button
#define BAND    8   // The pin used by band band push button
#define MODE    9   // The pin used by tune mode push button
#define MENU    10  // The pin used by tune menu push button
#define RX_TX   6   // The pin used by RX/TX selector switch
  //  RX = switch open, TX = switch closed to GND
#define TX_LED  1   // The pin used by TX LED output
#define SM      A0  // The pin used by Signal Meter A/D input

// On an Arduino Pro Mini: 10-bit ADC(5V)
//#define SM_GAIN 102 // Adjust the sensitivity of Signal Meter A/D input
  //  102 = 500mV, 205 = 1V, 307 = 1.5V, 409 = 2V, 512 = 2.5V, 1023 = 5V(max)
// On an Seeed Studio XIAO SAMD21: 12-bit ADC(3.3V)
#define SM_GAIN 4095  // Adjust the sensitivity of Signal Meter A/D input
#define SM_GAIN 1241  // 2025/08/17 Sun Version 1.17 modified
  //  620 = 500mV, 1241 = 1V, 1861 = 1.5V, 2481 = 2V, 3102 = 2.5V, 4095 = 3.3V(max)
#define SM_MIN  0     // Signal Meter(min)
#define SM_MAX  14    // Signal Meter(max)

#define FLAG_ON   1
#define FLAG_OFF  0

//********************************************************************
//* Variable Declaration（変数宣言）
//********************************************************************
// Create Rotary encoder object
Rotary r = Rotary(ENCODER_CLOCK_PIN, ENCODER_DATA_PIN);

// Create Si5351 module object
// Etherkit Si5351 Library
Si5351 si5351;

// Create display module object
//Adafruit_SSD1306 display(SCREEN_WIDTH, SCREEN_HEIGHT, &Wire, OLED_RESET);
Adafruit_SH1106G display(SCREEN_WIDTH, SCREEN_HEIGHT, &Wire, OLED_RESET);

// Step table struct
struct StepTable {
  String step_name; // Step name
  int step_value;   // Step value[Hz]
};
// Step table
const StepTable step_table[] = {
  {"   1Hz",       1},
  {"  10Hz",      10},
  {" 100Hz",     100},  // 2025/06/22 Sun Version 1.12 added
  {"  1kHz",    1000},
  {"  5kHz",    5000},
  {" 10kHz",   10000},
  {" 50kHz",   50000},  // 2025/06/22 Sun Version 1.12 added
  {"100kHz",  100000},  // 2025/06/22 Sun Version 1.12 added
  {"  1MHz", 1000000}
};
// Step table number of lines
int step_table_lines;

// Band plan table struct
struct BandPlanTable {
  String band_name; // Band name
  long start_freq;  // Start frequency[Hz]
  long stop_freq;   // Stop frequency[Hz]
};
// Band plan table(JARL band plan 20230925)
const BandPlanTable bandplan_table[] = {
//  {"2200",    135700,    137800},
//  {"600m",    472000,    479000},
//  {"160m",   1800000,   1875000},
//  {"160m",   1907500,   1912500},
//  {" 80m",   3500000,   3580000},
//  {" 80m",   3599000,   3612000},
//  {" 80m",   3662000,   3687000},
//  {" 75m",   3702000,   3716000},
//  {" 75m",   3745000,   3770000},
//  {" 75m",   3791000,   3805000},
  {" 40m",   7000000,   7200000},
  {" 30m",  10100000,  10150000},
  {" 20m",  14000000,  14350000},
  {" 17m",  18068000,  18168000},
  {" 15m",  21000000,  21450000},
  {" 12m",  24890000,  24990000},
  {" 10m",  28000000,  29700000},
  {"  6m",  50000000,  54000000},
  {"  2m", 144000000, 146000000},
//  {"70cm", 430000000, 440000000}
};
// Band plan table number of lines
int bandplan_table_lines;

// Mode table struct
struct ModeTable {
  String mode_name; // Mode name
  int if_value;     // IF value[Hz]
  int bfo_value;    // BFO value[Hz]
};
// Mode table
const ModeTable mode_table[] = {
  {"AM ", IF, AM_BFO},
  {"LSB", IF - (BANDWIDTH / 2), LSB_BFO}, // Upper side heterodyne 2025/07/03 Thu Version 1.15 modified
  {"USB", IF + (BANDWIDTH / 2), USB_BFO}, // Upper side heterodyne 2025/07/03 Thu Version 1.15 modified
  {"CW ", IF, CW_BFO}
};
// Mode table number of lines
int mode_table_lines;

// Store table struct
typedef struct {
  long cal_factor;    // Si5351 module calibration factor
  long default_freq;  // Default frequency
} StoreTable;

// Create FlashStorage object
// Reserve a portion of flash memory to store a "store_table" and call it "flash_store"
FlashStorage(flash_store, StoreTable);

StoreTable store_table;         // EEPROM alternative(Flash memory)
unsigned long freq;             // Frequency（周波数）
unsigned long freqold;          // Old frequency（前周波数）
unsigned long fstep;            // Frequency increase/decrease step（周波数増減幅）
long inter_freq;                // Intermediate frequency（中間周波数）
long inter_freqold = 0;         // Old Iintermediate frequency（前中間周波数）
long bfo_freq;                  // BFO(Beat Frequency Oscillator) frequency（うなり周波発振器周波数）
long cal_freq = 10000000;       // 10MHz
long cal_factor = SI5351_CAL_F; // Si5351 module calibration factor

unsigned int smval;             // Signal Meter Value
byte step_count;
byte band_count;
byte mode_count;
byte smrv;                      // Range value converted from signal（シグナルから変換されたレンジ値）
byte smrvold;                   // Old range value converted from signal（シグナルから変換された前レンジ値）
byte rx_tx_status = STATUS_RX;  // RX
unsigned int period = 100;
unsigned long time_now = 0; 

//********************************************************************
//* setup function
//********************************************************************
 void setup() {
  #ifdef DEBUG
    //  Serial initialization（シリアルを初期化する）
    Serial.begin(115200);
  #endif

  // I2C initialization（I2Cを初期化する）
  //Wire.begin();
  //Wire.setClock(400000);

  //display.begin(SSD1306_SWITCHCAPVCC, SCREEN_ADDRESS);
  display.begin(SCREEN_ADDRESS, true);
  //display.display();
  //delay(2000);
  display.clearDisplay();
  display.setTextColor(WHITE);
  // 2025/07/24 Thu Version 1.16 Go to after setting Si5351
  // start_screen_display();

  // Rotary encoder initialization（ロータリーエンコーダーを初期化する）
  r.begin(true, false);

  // Digital pin setting（デジタルピンを設定する）
  //pinMode(ENCODER_CLOCK_PIN, INPUT_PULLUP);
  //pinMode(ENCODER_DATA_PIN, INPUT_PULLUP);
  pinMode(STEP, INPUT_PULLUP);
  pinMode(BAND, INPUT_PULLUP);
  pinMode(MODE, INPUT_PULLUP);
  pinMode(MENU, INPUT_PULLUP);
  pinMode(RX_TX, INPUT_PULLUP);
  pinMode(TX_LED, OUTPUT);

  // Analog pin setting（アナログピンを設定する）
  pinMode(SM, INPUT);
  analogReadResolution(12); //Set to 12-bit resolution（12ビット分解能に設定する）

  // Read the content of "flash_store" into the "store_table" variable
  store_table = flash_store.read();
  // When the stored value is zero
  if (store_table.cal_factor == 0) {
    store_table.cal_factor = cal_factor;
  }

  // si5351 initialization（Si5351を初期化する）
  // Etherkit Si5351 Library
  si5351.init(SI5351_CRYSTAL_LOAD_8PF, 0, 0);             // 8pF
  si5351.set_correction(cal_factor * 100ULL, SI5351_PLL_INPUT_XO); // Calibration
  si5351.set_ms_source(SI5351_CLK0, SI5351_PLLA);         // CLK0(PLLA) RX BFO
  si5351.set_ms_source(SI5351_CLK1, SI5351_PLLA);         // CLK1(PLLA) TX OSC
  si5351.set_ms_source(SI5351_CLK2, SI5351_PLLB);         // CLK2(PLLB) RX LO
  //si5351.set_int(SI5351_CLK0, 0);
  //si5351.set_int(SI5351_CLK1, 0);
  //si5351.set_int(SI5351_CLK2, 0);

  // SI5351_CLK_DISABLE_LOW, SI5351_CLK_DISABLE_HIGH, SI5351_CLK_DISABLE_HI_Z, SI5351_CLK_DISABLE_NEVER
  si5351.set_clock_disable(SI5351_CLK0, SI5351_CLK_DISABLE_LOW);
  si5351.set_clock_disable(SI5351_CLK1, SI5351_CLK_DISABLE_LOW);
  si5351.set_clock_disable(SI5351_CLK2, SI5351_CLK_DISABLE_LOW);
  si5351.set_clock_pwr(SI5351_CLK0, 0);
  si5351.set_clock_pwr(SI5351_CLK1, 0);
  si5351.set_clock_pwr(SI5351_CLK2, 0);
  
  si5351.output_enable(SI5351_CLK0, 0);
  si5351.output_enable(SI5351_CLK1, 0);
  si5351.output_enable(SI5351_CLK2, 0);

  start_screen_display(); // 2025/07/24 Thu Version 1.16 moving on

  // Rotary encoder interrupt setting（ロータリーエンコーダー割り込みを設定する）
  attachInterrupt(digitalPinToInterrupt(ENCODER_DATA_PIN), REIF, CHANGE);
  attachInterrupt(digitalPinToInterrupt(ENCODER_CLOCK_PIN), REIF, CHANGE);

  // Set step table number of lines
  step_table_lines = sizeof(step_table) / sizeof(StepTable);
  // Set band table number of lines
  bandplan_table_lines = sizeof(bandplan_table) / sizeof(BandPlanTable);
  // Set mode table number of lines
  mode_table_lines = sizeof(mode_table) / sizeof(ModeTable);

  band_count = BAND_INIT;
  // Set band
  set_band();
  step_count = STEP_INIT;
  // Set step
  set_step();
  mode_count = MODE_INIT;
  // Set mode
  set_mode();
}

//********************************************************************
//* loop function
//********************************************************************
void loop() {
// Definitions（定義）
// 変数宣言（Variable Declaration）
// Variables（変数）
static bool menu_flag = FLAG_OFF; // Menu flag
static bool tx_flag = FLAG_OFF;   // TX flag

#ifdef DEBUG
  si5351.update_status();
  Serial.print("SYS_INIT: ");
  Serial.print(si5351.dev_status.SYS_INIT);
  Serial.print("  LOL_A: ");
  Serial.print(si5351.dev_status.LOL_A);
  Serial.print("  LOL_B: ");
  Serial.print(si5351.dev_status.LOL_B);
  Serial.print("  LOS: ");
  Serial.print(si5351.dev_status.LOS);
  Serial.print("  REVID: ");
  Serial.println(si5351.dev_status.REVID);
#endif

  // When frequency changes
  if (freqold != freq){
    time_now = millis();
    // Set frequency to Si5351
    tune_generate();
    freqold = freq;
  }

  // When intermediate frequency changes
  if (inter_freqold != inter_freq){
    time_now = millis();
    // Set frequency to Si5351
    tune_generate();
    inter_freqold = inter_freq;
  }

  // When range value change
  if (smrvold != smrv){
    time_now = millis();
    smrvold = smrv;
  }

  // When push STEP button
  if (digitalRead(STEP) == LOW){
    time_now = (millis() + 300);
    // Incremental step index
    inc_step_set();
    delay(300);
  }

  // When push BAND button
  if (digitalRead(BAND) == LOW){
    time_now = (millis() + 300);
    // Incremental band index
    inc_band_set();
    delay(300);
  }

  // When push MODE button 
  if (digitalRead(MODE) == LOW){
    time_now = (millis() + 300);
    // Incremental mode index
    inc_mode_set();
    delay(300);
  }

  // When push PTT
  if (digitalRead(RX_TX) == LOW){
    // When it is not AM
    if (mode_count != MODE_AM){
      rx_tx_status = STATUS_OUT_MODE; // MODE out of range
      //Display frequency
      display_frequency();
      // Display layout
      layout(VFO_MODE);
      delay(300);

    // Otherwise, when the frequency is not in the 6m band
    //} else if ((freq < bandplan_table[BAND_6M].start_freq) || (bandplan_table[BAND_6M].stop_freq < freq)){
    // Transmission frequency range change 2025/08/25 Mon Version 1.18 modified 
    } else if ((freq < bandplan_table[BAND_6M].start_freq + BANDWIDTH) || (bandplan_table[BAND_6M].stop_freq - BANDWIDTH < freq)){
      rx_tx_status = STATUS_OUT_TX; // TX out of range
      //Display frequency
      display_frequency();
      // Display layout
      layout(VFO_MODE);
      delay(300);

    // Otherwise 
    } else {
      rx_tx_status = STATUS_TX; // TX
      // Display frequency
      display_frequency();
      // Display layout
      layout(VFO_MODE);
      // Etherkit Si5351 Library
      si5351.set_clock_pwr(SI5351_CLK0, 0);
      si5351.output_enable(SI5351_CLK0, 0);
      si5351.set_clock_pwr(SI5351_CLK2, 0);
      si5351.output_enable(SI5351_CLK2, 0);
      Transmission();
    }

    rx_tx_status = STATUS_RX; // RX
    //Display frequency
    display_frequency();
    // Display layout
    layout(VFO_MODE);
    // Set mode
    set_mode(); 
    //delay(300);
  }

  // When push MENU button
  if (digitalRead(MENU) == LOW){
    // Set menu flag to ON
    menu_flag = FLAG_ON;
    delay(300);
  // Otherwise, when the menu flag is ON
  } else if (menu_flag == FLAG_ON){

    si5351.set_clock_pwr(SI5351_CLK0, 0);
    si5351.output_enable(SI5351_CLK0, 0);
    //si5351.set_clock_pwr(SI5351_CLK1, 0);
    //si5351.output_enable(SI5351_CLK1, 0);
    si5351.set_clock_pwr(SI5351_CLK2, 0);
    si5351.output_enable(SI5351_CLK2, 0);

    // Menu mode
    menu_mode();

    band_count = BAND_INIT;
    // Set band
    set_band();
    step_count = STEP_INIT;
    // Set step
    set_step();
    mode_count = MODE_INIT;
    // Set mode
    set_mode();

    // Set menu flag to OFF
    menu_flag = FLAG_OFF;
    delay(300);
  }

  // After the end of the period
  if ((time_now + period) > millis()){
    // Display frequency
    display_frequency();
    // Display layout
    layout(VFO_MODE);
  }

  // Read signal and convert range
  signal_read();
}

//********************************************************************
//* Functions（関数）
//********************************************************************
//********************************************************************
//* Rotary encoder interrupt function（ロータリーエンコーダー割り込み関数）
//*  Set rotary encoder interrupt（ロータリーエンコーダー割り込み関数を設定する）
//*   Clockwise:         DIR_CW 0x10
//*   Counter-clockwise: DIR_CCW 0x20
//********************************************************************
void REIF() {
  // Declare and retrieve the rotary encoder return value
  // （ロータリーエンコーダー戻り値を宣言し取得する）
  char result = r.process();

  // Return 1 when clockwise
  if (result == DIR_CW) set_frequency(DIR_CW);
  // Otherwise, return -1 when counter-clockwise
  else if (result == DIR_CCW) set_frequency(DIR_CCW);
}

//********************************************************************
//* Frequency setting function for Si5351（Si5351の周波数設定関数）
//*  Set frequency to Si5351（Si5351に周波数を設定する）
//********************************************************************
void tune_generate() {
  // Etherkit Si5351 Library
  // Set PLLA and PLLB to 800MHz for automatic tuning
  si5351.set_freq(freq * 100ULL, SI5351_CLK1);                // TX OSC
  si5351.set_freq((freq + inter_freq) * 100ULL, SI5351_CLK2); // RX LO
}

//********************************************************************
//* Frequency setting function（周波数設定関数）
//*  Set frequency with rotary encoder（ロータリーエンコーダーで周波数を設定する）
//********************************************************************
void set_frequency(short dir) {
  // When clockwise
  if (dir == DIR_CW){
    // Increment frequency by a step
    freq = freq + fstep;
  }
  // When the maximum frequency is exceeded 
  if (freq >= FREQ_MAX){
    // Set to maximum frequency
    freq = FREQ_MAX;
  }
  // When counter-clockwise
  if (dir == DIR_CCW){
    // When within the maximum frequency 
    if ((freq - fstep) <= FREQ_MAX){
      // Decrement the frequency by a step
      freq = freq - fstep;
    }
  }
  // When the step exceeds 1MHz at 1MHz
  if (fstep == 1000000 && freq <= 1000000){
    // Set to 1MHz
    freq = 1000000;
  // 2025/06/22 Sun Version 1.12 added
  // When the step exceeds 100kHz at 100kHz
  } else if (fstep == 100000 && freq <= 100000){  
    // Set to 100kMHz
    freq = 100000;
  // 2025/06/22 Sun Version 1.12 added
  // When the step exceeds 50kHz at 50kHz
  } else if (fstep == 50000 && freq <= 50000){
    // Set to 50kMHz
    freq = 50000;
  // Otherwise, when the minimum frequency is exceeded 
  } else if (freq < FREQ_MIN){
    // Set to minimum frequency
    freq = FREQ_MIN;
  }

  // 2025/06/12 Thu Version 1.10 added
  // When frequency is less than IF in LSB mode, change to USB mode
  // 2025/06/13 Fri Version 1.11 added
  // When frequency is less than IF in LSB mode, change to next mode
  // 2025/06/22 Sun Version 1.12 deleted
  //if (freq <= IF && mode_count == MODE_LSB) inc_mode_set();
}

//********************************************************************
//* Frequency display function（周波数表示関数）
//*  Display frequency（周波数を表示する）
//********************************************************************
void display_frequency() {
  unsigned int m = freq / 1000000;          // MHz unit
  unsigned int k = (freq % 1000000) / 1000; // kHz unit
  unsigned int h = (freq % 1000) / 1;       // Hz unit

  // Clear screen
  display.clearDisplay();
  display.setTextSize(2, 4);

  // When less than 1MHz
  if (m < 1){
    display.setCursor(44, 0);
    display.printf("%003d.%003d", k, h);
  // Others, when less than 100MHz
  } else if (m < 100){
    display.setCursor(8, 0);
    display.printf("%2d.%003d.%003d", m, k, h);
  // Others, when 100MHz or higher
  } else if (m >= 100){
    unsigned int h = (freq % 1000) / 10;
    display.setCursor(8, 0);
    display.printf("%2d.%003d.%02d", m, k, h);
  }
}

//********************************************************************
//* Step index increment function（ステップ索引増分関数）
//*  Incremental step index（ステップ索引を増分する）
//********************************************************************
void inc_step_set() {
  step_count++;
  if (step_count >= step_table_lines) step_count = 0;
  set_step();
}

//********************************************************************
//* Step setting function（ステップ設定関数）
//*  Set step（ステップを設定する）
//********************************************************************
void set_step() {
  fstep = step_table[step_count].step_value;
}

//********************************************************************
//* Band index increment function（バンド索引増分関数）
//*  Incremental band index（バンド索引を増分する）
//********************************************************************
void inc_band_set() {
  band_count++;
  if (band_count >= bandplan_table_lines) band_count = 0;
  set_band();
  delay(50);
}

//********************************************************************
//* Band setting function（バンド設定関数）
//*  Set band（バンドを設定する）
//********************************************************************
void set_band() {
  freq = bandplan_table[band_count].start_freq;
  // Etherkit Si5351 Library
  //si5351.pll_reset(SI5351_PLLA);
  //si5351.pll_reset(SI5351_PLLB);
  //step_count = 4;
  set_step();
  set_mode();
}

//********************************************************************
//* Mode index increment function（モード索引増分関数）
//*  Incremental mode index（モード索引を増分する）
//********************************************************************
void inc_mode_set() {
  mode_count++;
  if (mode_count >= mode_table_lines) mode_count = 0;

  // 2025/06/22 Sun Version 1.12 deleted
  // When frequency is less than IF in LSB mode
  //if (freq <= IF && mode_count == MODE_LSB) mode_count++;
  set_mode(); // 2025/06/13 Fri Version 1.11 added Modification of mode setting
  delay(50);
}

//********************************************************************
//* Mode setting function（モード設定関数）
//*  Set mode（モードを設定する）
//********************************************************************
void set_mode() {
  inter_freq = mode_table[mode_count].if_value;
  bfo_freq = mode_table[mode_count].bfo_value;

  // CLK0 RX BFO
  // Etherkit Si5351 Library
  si5351.set_clock_pwr(SI5351_CLK0, 0);
  //si5351.pll_reset(SI5351_PLLA);
  // Set PLLA and PLLB to 800MHz for automatic tuning
  si5351.set_freq(bfo_freq * 100ULL, SI5351_CLK0);
  si5351.drive_strength(SI5351_CLK0, SI5351_DRIVE_2MA);

  if (bfo_freq == 0){
    // Etherkit Si5351 Library
    si5351.set_clock_pwr(SI5351_CLK0, 0);
    si5351.output_enable(SI5351_CLK0, 0);
  } else {
    // Etherkit Si5351 Library
    si5351.set_clock_pwr(SI5351_CLK0, 1);
    si5351.output_enable(SI5351_CLK0, 1);
  }

  // CLK1 TX OSC
  // Etherkit Si5351 Library
  si5351.set_clock_pwr(SI5351_CLK1, 0);
  //si5351.pll_reset(SI5351_PLLA);
  // Set PLLA and PLLB to 800MHz for automatic tuning
  si5351.set_freq(freq * 100ULL, SI5351_CLK1);
  // 2025/06/13 Fri Version 1.11 modified
  //si5351.drive_strength(SI5351_CLK1, SI5351_DRIVE_2MA);
  // 2025/06/27 Fri Version 1.13 modified
  //si5351.drive_strength(SI5351_CLK1, SI5351_DRIVE_8MA);
  //si5351.drive_strength(SI5351_CLK1, SI5351_DRIVE_6MA);
  si5351.drive_strength(SI5351_CLK1, SI5351_DRIVE_4MA);
  si5351.set_clock_pwr(SI5351_CLK1, 0);
  si5351.output_enable(SI5351_CLK1, 0);

  // CLK2 RX LO
  // Etherkit Si5351 Library
  si5351.set_clock_pwr(SI5351_CLK2, 0);
  //si5351.pll_reset(SI5351_PLLB);
  // Set PLLA and PLLB to 800MHz for automatic tuning
  si5351.set_freq((freq + (inter_freq)) * 100ULL, SI5351_CLK2);
  si5351.drive_strength(SI5351_CLK2, SI5351_DRIVE_2MA);
  si5351.set_clock_pwr(SI5351_CLK2, 1);
  si5351.output_enable(SI5351_CLK2, 1);
}

//********************************************************************
//* Transmission function（送信関数）
//*  Transmit（送信する）
//********************************************************************
void Transmission() {
  // CLK1 TX OSC
  // Etherkit Si5351 Library
  si5351.set_clock_pwr(SI5351_CLK1, 1);
  si5351.output_enable(SI5351_CLK1, 1);
  while (digitalRead(RX_TX) == LOW){
    delay(300);
  }
}

//********************************************************************
//* Menu mode function（メニューモード関数）
//*  Process menu（メニュー処理をする）
//********************************************************************
void menu_mode() {
  // Definitions（定義）
  // 変数宣言（Variable Declaration）
  // Variables（変数）
  static bool menu_flag = FLAG_OFF;   // Menu flag
  static bool write_flag = FLAG_OFF;  // Write flag

  freq = cal_freq;
  freqold = freq;
  // CLK1 TX OSC
  // Etherkit Si5351 Library
  si5351.set_clock_pwr(SI5351_CLK1, 0);
  //si5351.pll_reset(SI5351_PLLA);
  // Set PLLA and PLLB to 800MHz for automatic tuning
  si5351.set_freq(freq * 100ULL, SI5351_CLK1);
  //si5351.drive_strength(SI5351_CLK1, SI5351_DRIVE_2MA);
  si5351.set_clock_pwr(SI5351_CLK1, 1);
  si5351.output_enable(SI5351_CLK1, 1);

  // Menu mode loop
  while (true){
    // When frequency changes
    if (freqold != freq){
      time_now = millis();
      // Set frequency to Si5351
      tune_generate();
      freqold = freq;
    }

    // When push STEP button
    if (digitalRead(STEP) == LOW){
      time_now = (millis() + 300);
      // Incremental step index
      inc_step_set();
      delay(300);
    }

    // Display frequency
    display_frequency();

    cal_factor = freq - cal_freq + store_table.cal_factor;

    // Display layout
    layout(MENU_MODE);

    // When push write(BAND) button
    if (digitalRead(BAND) == LOW){
      // Set write flag to ON
      write_flag = FLAG_ON;
      delay(300);
    // Otherwise, when the write flag is ON
    } else if (write_flag == FLAG_ON){
      // Write
      store_table.cal_factor = cal_factor;
      flash_store.write(store_table);

      si5351.set_correction(cal_factor * 100ULL, SI5351_PLL_INPUT_XO); // Calibration
      freq = cal_freq;
      freqold = freq;
      // CLK1 TX OSC
      // Etherkit Si5351 Library
      si5351.set_clock_pwr(SI5351_CLK1, 0);
      //si5351.pll_reset(SI5351_PLLA);
      // Set PLLA and PLLB to 800MHz for automatic tuning
      si5351.set_freq(freq * 100ULL, SI5351_CLK1);
      //si5351.drive_strength(SI5351_CLK1, SI5351_DRIVE_2MA);
      si5351.set_clock_pwr(SI5351_CLK1, 1);
      si5351.output_enable(SI5351_CLK1, 1);

      // Set write flag to OFF
      write_flag = FLAG_OFF;
    }

    // When push MENU button
    if (digitalRead(MENU) == LOW){
      // Set menu flag to ON
      menu_flag = FLAG_ON;
      delay(300);
    // Otherwise, when the menu flag is ON
    } else if (menu_flag == FLAG_ON){
      // Set menu flag to OFF
      menu_flag = FLAG_OFF;
      // Break from loop
      break;
    }
  }
}

//********************************************************************
//* Layout disply function（レイアウト表示関数）
//*  Display layout（レイアウトを表示する）
//********************************************************************
void layout(short display_mode) {
  display.setTextColor(WHITE);
  display.drawLine(0, 29, 106, 29, WHITE);
  display.drawLine(0, 46, 127, 46, WHITE);
  display.drawLine(68, 29, 68, 46, WHITE);  // 2025/06/22 Sun Version 1.12 modified
  display.drawLine(106, 29, 106, 46, WHITE);

  if (display_mode == VFO_MODE) {
    display.drawLine(24, 29, 24, 46, WHITE);  // 2025/06/22 Sun Version 1.12 modified
    display.drawLine(106, 46, 106, 63, WHITE);
    // Mode
    display.setTextSize(2);  
    display.setCursor(30, 31);  // 2025/06/22 Sun Version 1.12 modified
    display.print(mode_table[mode_count].mode_name);
  }

  // Step
  display.setTextSize(1, 2);
  display.setCursor(70, 31);  // 2025/06/22 Sun Version 1.12 modified
  display.print(step_table[step_count].step_name);

  // Step unit
  display.setCursor(110, 31);
  if (freq < 1000000) display.print("kHz");
  if (freq >= 1000000) display.print("MHz");

  // When VFO mode
  if (display_mode == VFO_MODE) {
    display.setCursor(110, 48);
    if (rx_tx_status == STATUS_RX){         // RX
      display.print("RX");
      digitalWrite(TX_LED, LOW);
    } else if (rx_tx_status == STATUS_TX){  // TX
      display.print("TX");
      digitalWrite(TX_LED, HIGH);
    } else {
      display.print("  ");
    }

    // Band
    display.setTextSize(1, 2);
    display.setCursor(0, 31);
    display.print(bandplan_table[band_count].band_name);

    display.invertDisplay(0);

    switch (rx_tx_status){
      case STATUS_TX:       // When TX
        display.setCursor(0, 48);
        display.print("  Busy (ON AIR)");
        break;

      case STATUS_OUT_TX:   // When TX is out of range
        display.setCursor(0, 48);
        display.invertDisplay(1);
        display.print(" TX out of range");
        break;

      case STATUS_OUT_MODE: // When MODE is out of range
        display.setCursor(0, 48);
        display.invertDisplay(1);
        display.print("MODE out of range");      
        break;

      default:
        display.drawLine(33, 58, 33, 63, WHITE);  // S3
        display.drawLine(45, 56, 45, 63, WHITE);  // S5
        display.drawLine(57, 54, 57, 63, WHITE);  // S7
        display.drawLine(69, 52, 69, 63, WHITE);  // S9
        draw_bargraph();
    }

  // Otherwise(Menu Mode)
  } else {
    // Calibration factor
    display.setTextSize(1, 2);
    display.setCursor(0, 31);
    display.printf("CAL=%d", cal_factor);

    // Function button
    display.setCursor(0, 48);
    display.print("Exit=MENU  Write=BAND");
  }
  display.display();
}

//********************************************************************
//* Signal read function（シグナル読み込み関数）
//*  Read signal and convert range（シグナルを読み込み、範囲を変換する）
//********************************************************************
void signal_read() {
  // Read signal
  smval = analogRead(SM);
  // Convert range
  smrv = map(smval, 0, SM_GAIN, SM_MIN, SM_MAX);
  //
  if (smrv > SM_MAX) smrv = SM_MAX;
}

//********************************************************************
//* Bargraph display function（バーグラフ表示関数）
//*  Display bargraph（バーグラフを表示する）
//********************************************************************
void draw_bargraph() {
  display.setTextSize(1, 2);

  // Bargraph
  display.setCursor(0, 48);
  display.print("SM");
  for (int i = 0; i < smrv; i++){
    display.fillRect(i * 6 + 16, 49, 5, 12, WHITE);
  }
}

//********************************************************************
//* Start screen display function（開始画面表示関数）
//*  Display start screen（開始画面を表示する）
//********************************************************************
void start_screen_display(){
  display.setTextSize(1);
  display.setCursor(0, 24);
  display.print("HHE-123 Si5351 VFO");
  display.setCursor(4, 32);
  display.print("JG1CCL V1R18 JH1YMC");
  display.display();
  delay(3000);
}
